// www.mm65.de (c) 2013

public class StartThreads {

    static final int THREADS_PER_DOT = 1000;

    public static void main(final String[] args) {
        new StartThreads(checkParams(args)).start();
    }

    private final StartParams startParams;
    private final Object lock = new Object();

    StartThreads(final StartParams params) {
        startParams = params;
    }

    private void start() {
        Thread[] threads = new Thread[startParams.threadCount];
        createThreads(threads, lock, startParams.busySeconds);
        startThreads(threads);
        terminateThreads(threads, lock);
        System.out.println("Ready.");
    }

    private static void createThreads(final Thread[] threads, final Object lock,
            final int busySeconds) {
        System.out.print("Creating " + threads.length
                + " threads (" + THREADS_PER_DOT + " threads per dot) ");
        for (int i = 0; i < threads.length; i++) {
            threads[i] = new Thread(new Runnable(){
                @Override public void run() {
                    try {
                        synchronized (lock) {
                            lock.wait();
                        }
                        Thread.sleep(busySeconds * 1000);
                    } catch (InterruptedException e) {
                    }
                }}, "MyThread_" + i);
            if (i % THREADS_PER_DOT == THREADS_PER_DOT - 1) {
                System.out.print(".");
            }
        }
        System.out.println();
    }

    private static void startThreads(Thread[] threads) {
        System.out.print("Starting " + threads.length + " threads ");
        for (int i = 0; i < threads.length; i++) {
            try {
                threads[i].start();
            } catch (OutOfMemoryError e) {
                System.err.println(" Problem while starting thread " + i
                        + ". Skipping starting further threads.");
                e.printStackTrace();
                break;
            }
            if (i % THREADS_PER_DOT == THREADS_PER_DOT - 1) {
                System.out.print(".");
            }
        }
        System.out.println();
    }

    private static void terminateThreads(final Thread[] threads, final Object lock) {
        System.out.print("Waiting on termination of " + threads.length + " threads ");
        int currentDots = 0;
        while (true) {
            notifyThreadsAndSleep(lock);
            int terminated = countTerminatedThreads(threads);
            currentDots = outputDots(currentDots, terminated);
            if (threads.length == terminated) {
                break;
            }
        }
        System.out.println();
    }

    private static void notifyThreadsAndSleep(final Object lock) {
        synchronized (lock) {
            lock.notifyAll();
        }
        try {
            Thread.sleep(1);
        } catch (InterruptedException e) {
        }
        synchronized (lock) {
            lock.notifyAll();
        }
    }

    private static int countTerminatedThreads(Thread[] threads) {
        int terminated = 0;
        for (int i = 0; i < threads.length; i++) {
            terminated += threads[i].isAlive() ? 0 : 1;
        }
        return terminated;
    }

    private static int outputDots(final int currentDots, final int terminated) {
        int newDots = terminated / THREADS_PER_DOT;
        for (int i = 0; i < newDots - currentDots; i++) {
            System.out.print(".");
        }
        return newDots;
    }

    private static StartParams checkParams(final String[] args) {
        StartParams params = new StartParams(args);
        if (params.error != null) {
            System.err.println(params.error);
            System.err.println();
            System.err.println(StartParams.showUsage());
            System.exit(1);
        }
        return params;
    }

}
